#!/bin/bash

PATH=/bin:/usr/bin:/usr/local/bin:/sbin:/usr/sbin:/usr/local/sbin:/usr/sfw/bin:$PATH
PS_CMD='ps auxww '
if [ -f "/etc/release" ]; then
	cat /etc/release | grep "Solaris 10" > /dev/null
	if [ $? = 0 ]; then
		PATH=/usr/xpg4/bin:/bin:/usr/bin:/usr/local/bin:/sbin:/usr/sbin:/usr/local/sbin:/usr/sfw/bin:$PATH
		PS_CMD='ps -e '
	fi
fi
OS_BINARY_TYPE=''
OS_NAME=`uname -s`
OS_ARCH=`uname -m`
LDD_VERSION=`ldd --version 2>/dev/null | awk 'NR==1{ print $NF }'`
SUCCESS=0
WARNING=1
FAILURE=2
BOOL_TRUE='True'
BOOL_FALSE='False'
IS_VENV_SUPPORT_NEEDED=$BOOL_FALSE
IS_ARM_PROCESSOR=$BOOL_FALSE
ECHO_PRINT=5
THIRTY_TWO_BIT='32-bit'
SIXTY_FOUR_BIT='64-bit'
SUDO=''
PRODUCT_NAME=ApplicationsManager
PRODUCT_NAME_UPPERCASE=`echo $PRODUCT_NAME | awk '{print toupper($0)}'`
PRODUCT_NAME_LOWERCASE=`echo $PRODUCT_NAME | awk '{print tolower($0)}'`
API_KEY=''
INSTALL_INPUT=''
INSTALL_PARAMS=''

IS_NONROOT='0'
INSTALL='INSTALL'
REINSTALL='REINSTALL'
UNINSTALL='UNINSTALL'

#local
LOCAL_SERVER=''

#livebuild
SERVER=__DOWNLOAD_HOST__
THIRTY_TWO_BIT_INSTALL_FILE=$PRODUCT_NAME"_Linux_32bit.install"
SIXTY_FOUR_BIT_INSTALL_FILE=$PRODUCT_NAME"_Linux_64bit.install"
SIXTY_FOUR_BIT_ARM_INSTALL_FILE="arm/"$PRODUCT_NAME"_Linux_64bit.install"
VENV_INSTALL_FILE=$PRODUCT_NAME"MonitoringAgent.install"
INSTALL_FILE=$PRODUCT_NAME"InstallScript.sh"

DOWNLOAD_URL=""

print_green() {
    printf "\033[32m%s\033[0m\n" "$*"
}

print_console() {
    printf "%s\n" "$*"
}

print_red() {
    printf "\033[31m%s\033[0m\n" "$*"
}


print_done() {
    print_green "Done"
}

`command -v which 2>/dev/null 1>/dev/null`
if [ $? = 0 ]; then
    COMMAND_CHECKER="which"
else
    COMMAND_CHECKER="command -v"
fi

if ! [ $(command -v tar) ]; then
	print_red "Tar utility not present to unzip product. Try installing tar before proceed with agent installation"
    exit $FAILURE
fi

log() {
	if [ "$1" = "$ECHO_PRINT" ]; then
		echo "$2"
	fi
}

getHardwarePlatform() {		
	if [ "`which file`" = "" ]; then		
	    if [ `/usr/bin/getconf LONG_BIT` = "64" ]; then
		    OS_BINARY_TYPE="$SIXTY_FOUR_BIT"
		elif [ `/usr/bin/getconf LONG_BIT` = "32" ]; then
		    OS_BINARY_TYPE="$THIRTY_TWO_BIT"
		fi
	else
		if /usr/bin/file /sbin/init | grep 'ELF 64-bit' >/dev/null; then
		    OS_BINARY_TYPE="$SIXTY_FOUR_BIT"
		elif /usr/bin/file /sbin/init | grep 'ELF 32-bit' >/dev/null; then
		    OS_BINARY_TYPE="$THIRTY_TWO_BIT"
		elif [ `/usr/bin/getconf LONG_BIT` = "64" ]; then
		    OS_BINARY_TYPE="$SIXTY_FOUR_BIT"
		elif [ `/usr/bin/getconf LONG_BIT` = "32" ]; then
		    OS_BINARY_TYPE="$THIRTY_TWO_BIT"
		fi
	fi	
}	

command_test() {
    $COMMAND_CHECKER $1 2>/dev/null 1>/dev/null
    if [ $? -ne 0 ]; then
    	if [ $2 = "optional" ]; then
    		WARNING_MSG=$WARNING_MSG" $1"
    		WARNING_FLAG=$BOOL_TRUE
    	else
    		ERROR_MSG=$ERROR_MSG" $1"
    		SEVERE_FLAG=$BOOL_TRUE
    	fi 
    fi
}

utilityCheck() {
	WARNING_MSG='Kindly install these if possible : '
	ERROR_MSG='Please install the following utility : '
	WARNING_FLAG=$BOOL_FALSE
	SEVERE_FLAG=$BOOL_FALSE
	command_test useradd "optional"
	command_test groupadd "optional"
	command_test usermod "optional"
	command_test awk "mandatory"
	command_test sed "mandatory"
 	if [ "$SEVERE_FLAG" = "$BOOL_TRUE" ]; then
 		print_red $ERROR_MSG
 		print_red $WARNING_MSG
 		exit $FAILURE
 	fi
}

detectArchitectureType(){
	if [[ "$OS_ARCH" = *"ppc64le"* ]] || [[ "$OS_ARCH" = *"s390x"* ]] ; then
	  IS_VENV_SUPPORT_NEEDED=$BOOL_TRUE
	fi
	if [[ "$OS_ARCH" = *"arm"* ]] || [[ "$OS_ARCH" = *"ARM"* ]] || [[ "$OS_ARCH" = *"Arm"* ]] || [[ "$OS_ARCH" = *"aarch"* ]] ; then
	  IS_ARM_PROCESSOR=$BOOL_TRUE
	fi
	print_console "Detected os arch : $OS_ARCH"
}

detectOs(){
	if [ "$OS_NAME" != "Linux" ]; then
		IS_VENV_SUPPORT_NEEDED=$BOOL_TRUE
	fi
	print_console "Detected OS : $OS_NAME"
}

format_version() {
        echo "$@" | awk -F. '{ printf("%03d%03d%03d\n", $1,$2,$3); }';
}

detectLibc(){
	libc_version="$(ldd --version 2>/dev/null | awk 'NR==1{ print $NF }')"
	if [ "$libc_version" != "" ]; then
		if [ "$(format_version $libc_version)" -lt "$(format_version 2.17)" ]; then
			print_red "Libc vesion $libc_version which is less than 2.17"
			IS_VENV_SUPPORT_NEEDED=$BOOL_TRUE
		fi
	fi
}

detectFlatCarOs(){
		OS_INFO=`uname -a`
        if [[ "$OS_INFO" = *"flatcar"* ]];then
                IS_VENV_SUPPORT_NEEDED=$BOOL_TRUE
        fi
}

checkShellUtility(){
	if command -v bash >/dev/null; then
    	SHELL_UTILITY="bash"
    else
    	SHELL_UTILITY="sh"
    fi
}

isRootUser() {
	if [ "$(id -u)" != "0" ]; then
		print_red "Please use 'sudo' or log in as root to execute the script"
		exit $FAILURE
	fi
}

isNonRootUser() {
	if [ "$(id -u)" == "0" ]; then
		print_red "Can't use -nonroot or -nr option when logged in as root"
		exit $FAILURE
	fi
}

checkForBinSupport(){
	checkShellUtility
	detectOs
	detectLibc
	detectArchitectureType 2>/dev/null
	detectFlatCarOs
}

setServerDomain() {
	if [ -n "$LOCAL_SERVER" ]; then
		SERVER=$LOCAL_SERVER
	elif [ "$PRODUCT_NAME_LOWERCASE" == "site24x7" ]; then
		DC="${MON_AGENT_API_KEY:0:2}"
		case "$DC" in 
			eu | cn | in | jp | uk | ca | sa ) 
				SERVER="${SERVER//.com/.$DC}"
			;;
			au )
				SERVER="${SERVER//.com/".net.au"}"
			;;
		esac
	elif [ -n "$HOST" ]; then
		PROTOCOL=`echo $HOST | awk -F :// '{print $1}' | grep "http"`
		if [ -z $PROTOCOL ]; then
			PROTOCOL="http"
			HOST=$PROTOCOL"://"$HOST
		fi
		SERVER=$HOST
	else
		echo "Please provide the server name in -host=<hostname> format"
		exit $FAILURE
	fi
}

setServerDomainForReinstall() {
	DC=$(echo "$SERVER_NAME" | awk -F. '{print $NF}')
	case "$DC" in 
		eu | cn | in | au | jp | uk | ca | sa ) 
			SERVER="${SERVER//.com/.$DC}"
		;;
		au )
			SERVER="${SERVER//.com/".net.au"}"
		;;
	esac
}

download_utility_check(){
	if [ $(command -v wget) ]; then
		if [ "$SSL_NO_CHECK" = "$BOOL_TRUE" ]; then
			SSL_DISABLE_PARAM="--no-check-certificate"
		fi
		DOWNLOAD_CMD="wget -O"
		print_console "wget detected"

	elif [ $(command -v curl) ]; then
		if [ "$SSL_NO_CHECK" = "$BOOL_TRUE" ]; then
			SSL_DISABLE_PARAM="-k"
		fi
		DOWNLOAD_CMD="curl -Lo"
		print_console "curl detected"

	elif [ $(command -v fetch) ]; then
		if [ "$SSL_NO_CHECK" = "$BOOL_TRUE" ]; then
			SSL_DISABLE_PARAM="--no-verify-peer"
		fi
		DOWNLOAD_CMD="fetch -o"
		print_console "fetch detected"
	else
		DOWNLOAD_CMD=''
		print_red "Couldn't detect curl, wget and fetch to download the installer. Hence exiting installation"
		exit $FAILURE
	fi
}

setDownloadUrl() {
	if [ -n "$DOWNLOAD_PATH" ]; then
		DWN_PATH=$DOWNLOAD_PATH
	else
		DWN_PATH="/server/"
	fi
	if [ "$OS_BINARY_TYPE" = "$SIXTY_FOUR_BIT" ]; then
		DOWNLOAD_FILE=$SIXTY_FOUR_BIT_INSTALL_FILE
	elif [ "$OS_BINARY_TYPE" = "$THIRTY_TWO_BIT" ]; then
		DOWNLOAD_FILE=$THIRTY_TWO_BIT_INSTALL_FILE	
	elif [ "$IS_ARM_PROCESSOR" = "$BOOL_TRUE" ]; then
		DOWNLOAD_FILE=$SIXTY_FOUR_BIT_ARM_INSTALL_FILE
	else
		DOWNLOAD_FILE="$VENV_INSTALL_FILE"
	fi
	DOWNLOAD_URL=$SERVER$DWN_PATH$DOWNLOAD_FILE
	print_console "Download url : $DOWNLOAD_URL"
}

deleteInstaller() {
        fileToDelete=`echo "${DOWNLOAD_URL##*/}"`
		if [ -f $fileToDelete  ];then
        	rm -f $fileToDelete
		fi
}

installAgent() {
	if [ "$IS_NONROOT" == "1" ]; then
		isNonRootUser
	else
		isRootUser
	fi
	getHardwarePlatform
	utilityCheck
	checkForBinSupport
	if [ "$INSTALL_INPUT" = "$REINSTALL" ];then
		if [ "$IS_NONROOT" == "1" ]; then
			username=`id -u -n`
			USER_HOME=`getent passwd $username | cut -d: -f6`
			INSTALL_DIR=$USER_HOME
		else
			INSTALL_DIR="/opt"
		fi
		PRODUCT_HOME=$INSTALL_DIR/$PRODUCT_NAME_LOWERCASE
		MON_AGENT_HOME=$PRODUCT_HOME/monagent
		SERVER_CONF=$MON_AGENT_HOME/conf/monagent.cfg
		SERVER_NAME=`cat $SERVER_CONF | grep -w "server_name" | head -1`
		setServerDomainForReinstall
	else
		setServerDomain
	fi
	download_utility_check
	setDownloadUrl
	echo ""
	echo "      -----------------------------------------------------------------------------------------------------------------------------------------------------"
	echo "      |																			  |"	
	echo "      |							      $PRODUCT_NAME_LOWERCASE server monitoring agent installation					         |"
	echo "      |																			  |"
	echo "      -----------------------------------------------------------------------------------------------------------------------------------------------------"
	echo ""	
	echo ""
	echo ""	
	if [ "$IS_VENV_SUPPORT_NEEDED" = "$BOOL_TRUE" ]; then
        echo "      --------------------------------------------------------Downloading install file for "$VENV_INSTALL_FILE"----------------------------------------------------------"  
 	else
        echo "      --------------------------------------------------------Downloading install file for "$OS_BINARY_TYPE"----------------------------------------------------------"
	fi 	
	echo ""	
	if [ "$IS_VENV_SUPPORT_NEEDED" = "$BOOL_TRUE" ]; then
       if [ -f $VENV_INSTALL_FILE  ];then
       		echo $VENV_INSTALL_FILE "already exists so removing and downloading new one"
       		rm -f $VENV_INSTALL_FILE
       fi
       $DOWNLOAD_CMD $VENV_INSTALL_FILE $DOWNLOAD_URL $SSL_DISABLE_PARAM
        if [ ! -f $VENV_INSTALL_FILE ]; then
           echo $VENV_INSTALL_FILE" not present hence quitting"
           exit $FAILURE
        fi
        if [[ "$OS_NAME" = *"Sun"* ]] || [[ "$OS_NAME" = *"AIX"* ]];then
             eval "$SHELL_UTILITY $VENV_INSTALL_FILE $INSTALL_PARAMS"
        else
             eval "$SHELL_UTILITY $VENV_INSTALL_FILE $INSTALL_PARAMS"
        fi
    elif [ "$OS_BINARY_TYPE" = "$THIRTY_TWO_BIT" ]; then
        if [ -f $THIRTY_TWO_BIT_INSTALL_FILE  ];then
       		echo $THIRTY_TWO_BIT_INSTALL_FILE "already exists so removing and downloading new one"
       		rm -f $THIRTY_TWO_BIT_INSTALL_FILE
       	fi
        $DOWNLOAD_CMD $THIRTY_TWO_BIT_INSTALL_FILE $DOWNLOAD_URL $SSL_DISABLE_PARAM
        if [ ! -f $THIRTY_TWO_BIT_INSTALL_FILE  ]; then
        echo $THIRTY_TWO_BIT_INSTALL_FILE" not present hence quitting"
        exit $FAILURE
        fi
        chmod 755 $THIRTY_TWO_BIT_INSTALL_FILE
        eval "$SHELL_UTILITY ./$THIRTY_TWO_BIT_INSTALL_FILE $INSTALL_PARAMS"
    else
        if [ -f $SIXTY_FOUR_BIT_INSTALL_FILE  ];then
       		echo $SIXTY_FOUR_BIT_INSTALL_FILE "already exists so removing and downloading new one"
       		rm -f $SIXTY_FOUR_BIT_INSTALL_FILE
       	fi
        $DOWNLOAD_CMD $SIXTY_FOUR_BIT_INSTALL_FILE $DOWNLOAD_URL $SSL_DISABLE_PARAM
        if [ ! -f $SIXTY_FOUR_BIT_INSTALL_FILE  ]; then
        echo $SIXTY_FOUR_BIT_INSTALL_FILE" not present hence quitting"
        exit $FAILURE
        fi
        chmod 755 $SIXTY_FOUR_BIT_INSTALL_FILE
        eval "$SHELL_UTILITY ./$SIXTY_FOUR_BIT_INSTALL_FILE $INSTALL_PARAMS"
    fi
    deleteInstaller
}

usage() {
	log $ECHO_PRINT ""
	log $ECHO_PRINT "Usage :"
	log $ECHO_PRINT "	Install command   : bash $INSTALL_FILE -i -automation=true -key=<device_key>"
	log $ECHO_PRINT ""	
	log $ECHO_PRINT "Options:"
	log $ECHO_PRINT "	-ri                          	Reinstall Server Agent with new source packages and old configuration"
	log $ECHO_PRINT "	-f                           	Force install the agent even when the agent present already"
	log $ECHO_PRINT "	-nr,-nonroot                 	Run the agent as a non-root user"
	log $ECHO_PRINT "	-nosyslog|-ns                	Disable syslog Monitoring"
	log $ECHO_PRINT "	-newkey|-nk                  	Register agent with new key"
	log $ECHO_PRINT "	-h                           	output usage information"
	log $ECHO_PRINT "	-ct,-CT                      	Configuration Template to be associated with the server"
	log $ECHO_PRINT "	-rule                        	Configuration Rule to be associated with the server"
	log $ECHO_PRINT "	-uid                         	Option to create a $PRODUCT_NAME_LOWERCASE agent User with a specific User ID"
	log $ECHO_PRINT "	-gid                         	Option to create a $PRODUCT_NAME_LOWERCASE agent group with a specific Group ID"
	log $ECHO_PRINT "	-dn,-DN                      	Display name of the server"
	log $ECHO_PRINT "	-gn,-GN                     	Group to which the server has to be added"
	log $ECHO_PRINT "	-tp,-TP                      	Threshold profile to be associated with the server"
	log $ECHO_PRINT "	-np,-NP                      	Notification profile to be associated with the server"
	log $ECHO_PRINT "	-rp,-RP                      	Resource Profile to be associated with the server"
	log $ECHO_PRINT "	-lp,-LP                      	Log Profile to be associated with the server"
	log $ECHO_PRINT "	-lt,-LT                      	Log Type to be associated with the server. For example -lt=SysLog"
	log $ECHO_PRINT "	-lf,-LF                    	Log Files to be collected for given log type. For example -lf=/var/log/syslog"
	log $ECHO_PRINT "	-automation=false            	Disable IT Automation Module in the server agent"
	log $ECHO_PRINT "	-plugins=false               	Disable Plugin Module in the server agent"
	log $ECHO_PRINT "	-location=<value>                        Add location tag to the server monitor"
	log $ECHO_PRINT "	-mysql='<mysql_configuration>'           Integrate MySQL Monitor along with Agent Installation"
	log $ECHO_PRINT "	-prometheus='<prometheus_configuration>' Integrate Prometheus along with Agent Installation"
	log $ECHO_PRINT "	-statsd='<statsd_configuration>'         Integrate Statsd along with Agent Installation"
	log $ECHO_PRINT "	-proxy                       	Set proxy to connect to the $PRODUCT_NAME_LOWERCASE server, if needed.
					EXAMPLE: -proxy=username:password@host:port, if there is no username and passowrd for proxy server then use -proxy=host:port"
	if [ "$PRODUCT_NAME" == "Site24x7" ]; then
		log $ECHO_PRINT "	-s24x7-agent                 	Run the agent as a site24x7-agent user"
		log $ECHO_PRINT ""
		log $ECHO_PRINT "For more command line installation configurations, refer more parameters at https://www.site24x7.com/help/admin/adding-a-monitor/command-line-installation.html"
		log $ECHO_PRINT ""
	fi
	exit $FAILURE
}

main() {
	while [ "$1" != "" ]; do
		KEY=`echo $1 | awk -F= '{print $1}'`
		VALUE=`echo $1 | awk -F= '{print $2}'`
		if [ "$VALUE" != "" ]; then
			ARGUMENT=$KEY'='\"$VALUE\"
		else
			ARGUMENT=$KEY
		fi
		INSTALL_PARAMS=$INSTALL_PARAMS' '$ARGUMENT
		case $KEY in
				-h|--help)
					usage
				;;
				-nr|-nonroot)
					IS_NONROOT='1'
				;;
				-i)
					INSTALL_INPUT=$INSTALL
				;;
				-ri)
					INSTALL_INPUT=$REINSTALL
				;;
				-u)
					INSTALL_INPUT=$UNINSTALL
				;;
				-key)
					MON_AGENT_API_KEY=$VALUE
				;;
				-host)
					HOST=$VALUE
				;;
				-dpath)
					DOWNLOAD_PATH=$VALUE
				;;
				-nocheckcert)
					SSL_NO_CHECK=$BOOL_TRUE
				;;
		esac
		shift
	done
	install_agent=$BOOL_TRUE
	if [ "$INSTALL_INPUT" = "$REINSTALL" ]; then
		install_agent=$BOOL_TRUE
		printf "Reinstallation in Progress \n"
	elif [ "$MON_AGENT_API_KEY" = "" ] || [ "$INSTALL_INPUT" = "" ]; then
		install_agent=$BOOL_FALSE
	fi
	if [ "$install_agent" = "$BOOL_TRUE" ]; then
		installAgent
		if [ "$OS_NAME" = "Linux" ]; then
			printf "\n"
			if [ "$PRODUCT_NAME" == "Site24x7" ]; then
				print_green "Have more servers? Try our bulk installation techniques. Refer link : https://www.site24x7.com/help/admin/adding-a-monitor/linux-server-monitoring.html#bulk-installation"
			fi
			printf "\n"
		fi
	else
		usage
	fi
}

main "$@"
